<?php
/**
 * Plugin Name: BangoPayBD Gateway
 * Plugin URI: https://wordpress.org/plugins/bangopaybd
 * Description: This plugin allows your customers to pay with Bkash, Nagad, Rocket, and all BD gateways via bangopaybd
 * Author: DEV ANOY
 * Author URI: https://bangopaybd.com
 * Version: 1.0.0
 * Requires at least: 6.2
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: bangopaybd
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

add_action('before_woocommerce_init', function () {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('cart_checkout_blocks', __FILE__, true);
    }
});

/**
 * Initialize the BangoPayBD gateway class.
 */
add_action('plugins_loaded', 'bangopaybd_init_gateway_class', 0);

/**
 * Initialize the BangoPayBD payment gateway.
 *
 * This function checks if the WooCommerce plugin is active before initializing
 * the BangoPayBD gateway class.
 */
function bangopaybd_init_gateway_class()
{
    if (!class_exists('WC_Payment_Gateway')) {
        return;
    }

    /**
     * WC_BangoPayBD_Gateway Class.
     *
     * Extends WC_Payment_Gateway to provide the BangoPayBD payment gateway.
     */
    class WC_BangoPayBD_Gateway extends WC_Payment_Gateway
    {

        private $apikey;
        private $currency_rate;
        private $is_digital;
        private $payment_site;
        private static $instance = null;

        public static function get_instance()
        {
            if (null === self::$instance) {
                self::$instance = new self();
            }
            return self::$instance;
        }
        
        public function __construct()
        {
            $this->id = 'bangopaybd';
            $this->icon = apply_filters('woocommerce_bangopaybd_icon', plugins_url('assets/logo.png', __FILE__));
            $this->has_fields = false;
            $this->method_title = __('BangoPayBD', 'bangopaybd');
            $this->method_description = __('Pay with Bkash, Nagad, Rocket, and all BD gateways via BangoPayBD.', 'bangopaybd');

            $this->supports = ['products'];

            // Load the settings.
            $this->init_form_fields();
            $this->init_settings();

            // Define user set variables.
            $this->title = $this->get_option('title');
            $this->description = $this->get_option('description');
            $this->enabled = $this->get_option('enabled');
            $this->apikey = $this->get_option('apikey');
            $this->currency_rate = $this->get_option('currency_rate');
            $this->is_digital = $this->get_option('is_digital') === 'yes';
            $this->payment_site = $this->get_option('payment_site', 'https://pay.bangopaybd.com');

            // Actions.
            add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
            add_action('woocommerce_api_wc_gateway_' . $this->id, [$this, 'handle_webhook']);

            if (isset($_GET['success1'])) {
                $order = new WC_Order($_GET['success1']);
                $this->update_order_status($order);
                wp_redirect($this->get_return_url($order));
                exit;
            }
        }

        /**
         * Initialize Gateway Settings Form Fields.
         *
         * Sets the values for the settings fields as displayed on the admin page.
         */
        public function init_form_fields()
        {
            $this->form_fields = [
                'enabled' => [
                    'title' => __('Enable/Disable', 'bangopaybd'),
                    'type' => 'checkbox',
                    'label' => __('Enable BangoPayBD', 'bangopaybd'),
                    'description' => __('Enable or disable the BangoPay payment gateway.', 'bangopaybd'),
                    'default' => 'no',
                ],
                'title' => [
                    'title' => __('Title', 'bangopaybd'),
                    'type' => 'text',
                    'description' => __('This controls the title which the user sees during checkout.', 'bangopaybd'),
                    'default' => __('BangoPayBD Gateway', 'bangopaybd'),
                    'desc_tip' => true,
                ],
                'description' => [
                    'title' => __('Description', 'bangopaybd'),
                    'type' => 'textarea',
                    'description' => __('This controls the description which the user sees during checkout.', 'bangopaybd'),
                    'default' => __('Pay with Bkash, Nagad, Rocket, and all BD gateways via BangoPayBD.', 'bangopaybd'),
                ],
                'apikey' => [
                    'title' => __('Enter API Key', 'bangopaybd'),
                    'type' => 'text',
                    'description' => __('Enter your BangoPayBD API Key.', 'bangopaybd'),
                    'default' => '##################',
                    'desc_tip' => true,
                ],
                'currency_rate' => [
                    'title' => __('Enter USD Rate', 'bangopaybd'),
                    'type' => 'number',
                    'description' => __('Enter the exchange rate for USD to BDT. Only applicable if your store currency is not BDT.', 'bangopaybd'),
                    'default' => '120',
                    'desc_tip' => true,
                ],
                'is_digital' => [
                    'title' => __('Enable Digital Product', 'bangopaybd'),
                    'label' => __('Enable this if you are primarily selling digital products.', 'bangopaybd'),
                    'type' => 'checkbox',
                    'description' => __('If enabled, orders will be marked as completed immediately after successful payment.', 'bangopaybd'),
                    'default' => 'no',
                ],
                'payment_site' => [
                    'title' => __('Payment Site URL', 'bangopaybd'),
                    'type' => 'hidden',
                    'description' => __('The base URL for the BangoPayBD payment gateway.', 'bangopaybd'),
                    'default' => 'https://pay.bangopaybd.com',
                    'desc_tip' => false,
                ],
            ];
        }

        /**
         * Process the payment and redirect to the payment gateway.
         *
         * @param int $order_id Order ID.
         * @return array|void
         */
        public function process_payment($order_id)
        {
            $order = wc_get_order($order_id);
            if (!$order) {
                wc_add_notice(__('Order not found.', 'bangopaybd-gateway'), 'error');
                return ['result' => 'fail'];
            }

            $data = [
                'api_key' => $this->apikey,
                'cus_name' => sanitize_text_field($order->get_billing_first_name()),
                'cus_email' => sanitize_email($order->get_billing_email()),
                'amount' => $order->get_total(),
                'success_url' => wc_get_page_permalink('checkout') . '?success1=' . $order->get_id(),
                'cancel_url' => wc_get_page_permalink('checkout')
            ];

            $header = [
                'url' => trailingslashit($this->payment_site) . 'api/payment/create',
            ];

            $response = $this->create_payment($data, $header);
            $response_data = json_decode($response, true);

            if (isset($response_data['payment_url'])) {
                return [
                    'result' => 'success',
                    'redirect' => $response_data['payment_url'],
                ];
            } else {
                wc_add_notice(__('Payment could not be initiated. Please try again or use a different payment method.', 'bangopaybd'), 'error');
                return [
                    'result' => 'failure',
                ];
            }
        }

        /**
         * Call the BangoPayBD API to create a payment request.
         *
         * @param array $data Payment data.
         * @param array $header API headers.
         * @return string|WP_Error API response.
         */
        private function create_payment($data, $header)
        {
            $url = $header['url'];
            $args = [
                'body' => http_build_query($data),
                'headers' => ['Content-Type' => 'application/x-www-form-urlencoded'],
                'method' => 'POST',
                'timeout' => 45,
            ];

            $response = wp_remote_post($url, $args);

            if (is_wp_error($response)) {
                error_log('BangoPayBD API Error: ' . print_r($response, true));
                return $response;
            }

            return wp_remote_retrieve_body($response);
        }

        /**
         * Handle the webhook from BangoPayBD (if needed for asynchronous notifications).
         *
         * @return void
         */
        public function handle_webhook()
        {
            // Implement webhook handling if BangoPayBD sends asynchronous payment notifications.
            // This would typically involve verifying the data received and updating the order status accordingly.
            // For simple redirect-based gateways, this might not be strictly necessary.
        }

        /**
         * Update the order status based on the successful payment.
         *
         * @param WC_Order $order The WooCommerce order object.
         * @return void
         */
        public function update_order_status($order)
        {
            if (isset($_GET['transactionId'])) {
                $transactionId = $_GET['transactionId'];
                $verification_data = [
                    'api_key' => $this->apikey,
                    'transaction_id' => $transactionId,
                ];
                $header = [
                    "url"     => $this->get_option('payment_site') . '/api/payment/verify',
                ];

                $response = $this->create_payment($verification_data, $header);
                $data = json_decode($response, true);

                if ($order->get_status() != 'completed') {
                    if ($data['status'] == "COMPLETED") {
                        $transaction_id = $data['transaction_id'];
                        $amount = $data['amount'];
                        $sender_number = $data['cus_email'];
                        $payment_method = 'BangoPayBD';
                        if ($this->get_option('is_digital') === 'yes') {
                            $order->update_status('completed', __("You Payment is complete by BangoPayBD", 'bangopaybd'));
                            // Reduce stock levels
                            $order->reduce_order_stock();
                            $order->add_order_note(__('Payment completed via BangoPayBD', 'bangopaybd'));
                            $order->payment_complete();
                        } else {
                            $order->update_status('processing', __("BangoPayBD payment was successfully processed.", 'bangopaybd'));
                            // Reduce stock levels
                            $order->reduce_order_stock();
                            $order->payment_complete();
                        }
                        return true;
                    } else {
                        $this->msg['class'] = 'error';
                        $this->msg['message'] = 'Payment verification failed';
                        $order->update_status('on-hold', __('Payment verification failed', 'bangopaybd'));
                    }
                }
            }
        }
    }
    
    // Define server-side integration for blocks
    class WC_BangoPayBD_Blocks extends \Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType
    {
        protected $name = 'bangopaybd';

        public function initialize()
        {
            $this->settings = get_option('woocommerce_bangopaybd_settings', []);
        }

        public function is_active()
        {
            return !empty($this->settings['enabled']) && 'yes' === $this->settings['enabled'];
        }

        public function get_payment_method_script_handles()
        {
            wp_register_script(
                'bangopaybd-blocks',
                plugins_url('assets/js/bangopaybd-blocks.js', __FILE__),
                ['wc-blocks-registry', 'wc-settings', 'wp-element', 'wp-i18n'],
                filemtime(plugin_dir_path(__FILE__) . 'assets/js/bangopaybd-blocks.js'),
                true
            );
            return ['bangopaybd-blocks'];
        }

        public function get_payment_method_data()
        {
            $gateway = WC_BangoPayBD_Gateway::get_instance();
            return [
                'title' => $gateway->title,
                'description' => $gateway->description,
                'icon' => $gateway->icon,
                'supports' => ['products'],
            ];
        }
    }

    add_action('woocommerce_blocks_payment_method_type_registration', function ($registry) {
        $registry->register(new WC_BangoPayBD_Blocks());
    });

    // Add filter to include payment gateway
    add_filter('woocommerce_payment_gateways', function ($gateways) {
        $gateways[] = 'WC_BangoPayBD_Gateway';
        return $gateways;
    });
}
